//
//  MetricRow.swift
//  death_app Watch App
//
//  Created by Task-Master AI
//

import SwiftUI

struct MetricRow: View {
    let title: String
    let value: String
    let subtitle: String?
    let icon: String
    let color: Color
    let trend: TrendDirection?
    
    init(
        title: String,
        value: String,
        subtitle: String? = nil,
        icon: String,
        color: Color = .blue,
        trend: TrendDirection? = nil
    ) {
        self.title = title
        self.value = value
        self.subtitle = subtitle
        self.icon = icon
        self.color = color
        self.trend = trend
    }
    
    var body: some View {
        HStack(spacing: 12) {
            iconView
            
            VStack(alignment: .leading, spacing: 2) {
                titleView
                valueView
                
                if let subtitle = subtitle {
                    subtitleView(subtitle)
                }
            }
            
            Spacer()
            
            if let trend = trend {
                trendIndicator(trend)
            }
        }
        .padding(.vertical, 8)
        .padding(.horizontal, 4)
        .accessibilityElement(children: .combine)
        .accessibilityLabel(accessibilityLabel)
    }
    
    private var iconView: some View {
        Image(systemName: icon)
            .font(.title3)
            .foregroundColor(color)
            .frame(width: 24, height: 24)
            .accessibilityHidden(true)
    }
    
    private var titleView: some View {
        Text(title)
            .font(.caption)
            .foregroundColor(.secondary)
            .lineLimit(1)
    }
    
    private var valueView: some View {
        Text(value)
            .font(.body.bold())
            .foregroundColor(.primary)
            .lineLimit(1)
    }
    
    private func subtitleView(_ text: String) -> some View {
        Text(text)
            .font(.caption2)
            .foregroundColor(.secondary)
            .lineLimit(2)
            .fixedSize(horizontal: false, vertical: true)
    }
    
    private func trendIndicator(_ trend: TrendDirection) -> some View {
        HStack(spacing: 2) {
            Image(systemName: trend.icon)
                .font(.caption)
                .foregroundColor(trend.color)
            
            Text(trend.rawValue.capitalized)
                .font(.caption2)
                .foregroundColor(trend.color)
        }
        .accessibilityLabel("Trend: \(trend.rawValue)")
    }
    
    private var accessibilityLabel: String {
        var label = "\(title): \(value)"
        
        if let subtitle = subtitle {
            label += ", \(subtitle)"
        }
        
        if let trend = trend {
            label += ", trend is \(trend.rawValue)"
        }
        
        return label
    }
}

// MARK: - Convenience Initializers

extension MetricRow {
    static func bestDay(
        date: Date,
        lifeExpectancy: Double,
        factors: [String]
    ) -> MetricRow {
        let formatter = DateFormatter()
        formatter.dateStyle = .medium
        
        let value = formatLifeExpectancy(lifeExpectancy)
        let subtitle = factors.isEmpty ? 
            formatter.string(from: date) : 
            "\(formatter.string(from: date)) • \(factors.prefix(2).joined(separator: ", "))"
        
        return MetricRow(
            title: "Best Day",
            value: value,
            subtitle: subtitle,
            icon: "star.fill",
            color: .green,
            trend: .improving
        )
    }
    
    static func worstDay(
        date: Date,
        lifeExpectancy: Double,
        factors: [String]
    ) -> MetricRow {
        let formatter = DateFormatter()
        formatter.dateStyle = .medium
        
        let value = formatLifeExpectancy(lifeExpectancy)
        let subtitle = factors.isEmpty ? 
            formatter.string(from: date) : 
            "\(formatter.string(from: date)) • \(factors.prefix(2).joined(separator: ", "))"
        
        return MetricRow(
            title: "Worst Day",
            value: value,
            subtitle: subtitle,
            icon: "exclamationmark.triangle.fill",
            color: .red,
            trend: .declining
        )
    }
    
    static func averageTrend(
        average: Double,
        timeframe: HistoricalTimeframe,
        trend: TrendDirection
    ) -> MetricRow {
        let value = formatLifeExpectancy(average)
        let subtitle = "Over \(timeframe.displayName.lowercased())"
        
        return MetricRow(
            title: "Average",
            value: value,
            subtitle: subtitle,
            icon: "chart.line.uptrend.xyaxis",
            color: trend.color,
            trend: trend
        )
    }
    
    static func totalChange(
        change: Double,
        timeframe: HistoricalTimeframe
    ) -> MetricRow {
        let isPositive = change >= 0
        let value = "\(isPositive ? "+" : "")\(String(format: "%.1f", change))y"
        let subtitle = "Total change over \(timeframe.displayName.lowercased())"
        
        return MetricRow(
            title: "Net Change",
            value: value,
            subtitle: subtitle,
            icon: isPositive ? "arrow.up.right" : "arrow.down.right",
            color: isPositive ? .green : .red,
            trend: isPositive ? .improving : .declining
        )
    }
    
    static func contributingFactor(
        factor: ContributingFactor
    ) -> MetricRow {
        let impact = factor.impact
        let isPositive = impact >= 0
        let value = "\(isPositive ? "+" : "")\(String(format: "%.2f", impact))y"
        
        return MetricRow(
            title: factor.name,
            value: value,
            subtitle: factor.description,
            icon: getFactorIcon(factorName: factor.name),
            color: isPositive ? .green : .red,
            trend: isPositive ? .improving : .declining
        )
    }
    
    private static func getFactorIcon(factorName: String) -> String {
        switch factorName.lowercased() {
        case let name where name.contains("heart"):
            return "heart.fill"
        case let name where name.contains("steps") || name.contains("activity"):
            return "figure.walk"
        case let name where name.contains("sleep"):
            return "bed.double.fill"
        case let name where name.contains("exercise") || name.contains("workout"):
            return "figure.run"
        case let name where name.contains("stress"):
            return "brain.head.profile"
        case let name where name.contains("weight"):
            return "scalemass.fill"
        default:
            return "chart.bar.fill"
        }
    }
    
    private static func formatLifeExpectancy(_ years: Double) -> String {
        let totalYears = Int(years)
        let months = Int((years - Double(totalYears)) * 12)
        
        if months > 0 {
            return "\(totalYears)y \(months)m"
        } else {
            return "\(totalYears)y"
        }
    }
}

#Preview {
    ScrollView {
        VStack(spacing: 12) {
            MetricRow.averageTrend(
                average: 75.4,
                timeframe: .week,
                trend: .improving
            )
            
            Divider()
            
            MetricRow.bestDay(
                date: Date().addingTimeInterval(-86400 * 2),
                lifeExpectancy: 76.2,
                factors: ["Great workout", "Excellent sleep"]
            )
            
            Divider()
            
            MetricRow.worstDay(
                date: Date().addingTimeInterval(-86400 * 5),
                lifeExpectancy: 74.8,
                factors: ["Poor sleep", "High stress"]
            )
            
            Divider()
            
            MetricRow.totalChange(
                change: 1.4,
                timeframe: .month
            )
            
            Divider()
            
            MetricRow.contributingFactor(
                factor: ContributingFactor(
                    name: "Heart Rate",
                    impact: 0.15,
                    description: "Improved resting heart rate contributing to longevity"
                )
            )
        }
        .padding()
    }
}